<?php
/**
 * Teste da API PIX Gateway
 * Este arquivo testa a conexão com a API usando os campos corretos
 */

// Verificar se estamos no WordPress
if (!defined('ABSPATH')) {
    echo "<h2>❌ Este arquivo deve ser executado no WordPress</h2>";
    exit;
}

// Verificar permissões
if (!current_user_can('manage_woocommerce')) {
    wp_die('Você não tem permissão para acessar esta página.');
}

echo "<h1>🧪 Teste da API PIX Gateway</h1>";
echo "<hr>";

// Obter configurações
$api_key = get_option('wc_pix_gateway_api_key', '');
$api_url = get_option('wc_pix_gateway_api_url', 'https://velozpay.online/api/v1/gateway/');

if (empty($api_key)) {
    echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ API Key não configurada!</div>";
    echo "<p>Configure a API Key nas configurações do plugin primeiro.</p>";
    exit;
}

if (empty($api_url)) {
    echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ URL da API não configurada!</div>";
    echo "<p>Configure a URL da API nas configurações do plugin primeiro.</p>";
    exit;
}

echo "<h3>📋 Configurações Atuais</h3>";
echo "<p><strong>API Key:</strong> " . substr($api_key, 0, 8) . "..." . substr($api_key, -4) . "</p>";
echo "<p><strong>API URL:</strong> {$api_url}</p>";

echo "<hr>";

// Testar conexão com dados completos
echo "<h3>🔌 Testando Conexão com a API</h3>";

$test_url = "https://velozpay.online//api/v1/adquirente/asaas/processa_pagamento.php";

// Dados de teste conforme a documentação
$test_data =array(
    "amount" => "5,10",
"cpf" => "394.480.048-69",
"email" => "c@gmail.com",
"name" => "caio teste",
"tipoPagamento" => "PIX",
"user_id" => "gomes"
);        

echo "<h4>📤 Dados sendo enviados:</h4>";
echo "<pre style='background: #f5f5f5; padding: 10px; border: 1px solid #ddd;'>";
echo json_encode($test_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
echo "</pre>";

echo "<h4>🌐 URL da API:</h4>";
echo "<p><code>{$test_url}</code></p>";

echo "<h4>🔑 Headers da requisição:</h4>";
echo "<ul>";
echo "<li><strong>Content-Type:</strong> application/json</li>";
echo "<li><strong>Authorization:</strong> Bearer " . substr($api_key, 0, 8) . "..." . substr($api_key, -4) . "</li>";
echo "<li><strong>Accept:</strong> application/json</li>";
echo "</ul>";

echo "<hr>";

// Fazer a requisição
echo "<h4>📡 Fazendo requisição...</h4>";

$response = wp_remote_post($test_url, array(
    'timeout' => 30,
    'headers' => array(
        'Content-Type' => 'application/json',
        'Authorization' => 'Bearer ' . $api_key,
        'Accept' => 'application/json'
    ),
    'body' => json_encode($test_data)
));

if (is_wp_error($response)) {
    echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ Erro na conexão:</div>";
    echo "<p><strong>Erro:</strong> " . $response->get_error_message() . "</p>";
    echo "<p><strong>Código:</strong> " . $response->get_error_code() . "</p>";
} else {
    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $headers = wp_remote_retrieve_headers($response);
    
    echo "<h4>📥 Resposta recebida:</h4>";
    echo "<p><strong>Status HTTP:</strong> <span style='color: " . ($status_code === 200 ? 'green' : 'red') . ";'>{$status_code}</span></p>";
    
    if ($status_code === 200) {
        echo "<div style='color: green; padding: 10px; border: 1px solid green;'>✅ Conexão bem-sucedida!</div>";
    } else {
        echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ Erro na API</div>";
    }
    
    echo "<h5>📄 Corpo da resposta:</h5>";
    echo "<pre style='background: #f5f5f5; padding: 10px; border: 1px solid #ddd; max-height: 300px; overflow-y: auto;'>";
    echo htmlspecialchars($body);
    echo "</pre>";
    
    // Tentar decodificar JSON
    $json_data = json_decode($body, true);
    if ($json_data) {
        echo "<h5>🔍 Resposta decodificada:</h5>";
        echo "<pre style='background: #f0f8ff; padding: 10px; border: 1px solid #87ceeb;'>";
        print_r($json_data);
        echo "</pre>";
        
        if (isset($json_data['status']) && $json_data['status'] === 'success') {
            echo "<div style='color: green; padding: 10px; border: 1px solid green;'>🎉 API funcionando perfeitamente!</div>";
        } else {
            echo "<div style='color: orange; padding: 10px; border: 1px solid orange;'>⚠️ API respondeu, mas com erro</div>";
        }
    }
    
    echo "<h5>📋 Headers da resposta:</h5>";
    echo "<pre style='background: #f5f5f5; padding: 10px; border: 1px solid #ddd;'>";
    print_r($headers);
    echo "</pre>";
}

echo "<hr>";

// Verificar se há logs de erro
echo "<h3>📝 Logs de Erro (se houver):</h3>";

if (function_exists('wc_get_logger')) {
    $logger = wc_get_logger();
    $logs = $logger->get_log_files();
    
    $pix_logs = array();
    foreach ($logs as $log_file => $log_path) {
        if (strpos($log_file, 'pix') !== false || strpos($log_file, 'gateway') !== false) {
            $pix_logs[] = $log_file;
        }
    }
    
    if (!empty($pix_logs)) {
        echo "<p>Logs encontrados:</p>";
        echo "<ul>";
        foreach ($pix_logs as $log_file) {
            echo "<li>📄 {$log_file}</li>";
        }
        echo "</ul>";
        
        echo "<p><a href='" . admin_url('admin.php?page=wc-status&tab=logs') . "' target='_blank'>Ver logs completos</a></p>";
    } else {
        echo "<p>Nenhum log específico do PIX encontrado.</p>";
    }
} else {
    echo "<p>Logger do WooCommerce não disponível.</p>";
}

echo "<hr>";

// Recomendações
echo "<h3>💡 Recomendações</h3>";

if ($status_code === 400) {
    echo "<div style='color: orange; padding: 10px; border: 1px solid orange;'>⚠️ Erro 400 - Verifique se todos os campos obrigatórios estão sendo enviados</div>";
    echo "<p>Possíveis problemas:</p>";
    echo "<ul>";
    echo "<li>Campos obrigatórios ausentes na requisição</li>";
    echo "<li>Formato dos dados incorreto</li>";
    echo "<li>Validação da API falhou</li>";
    echo "</ul>";
}

if ($status_code === 401) {
    echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ Erro 401 - API Key inválida ou expirada</div>";
    echo "<p>Verifique se a API Key está correta e ativa.</p>";
}

if ($status_code === 403) {
    echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ Erro 403 - Acesso negado</div>";
    echo "<p>Verifique as permissões da sua conta na API.</p>";
}

if ($status_code === 500) {
    echo "<div style='color: red; padding: 10px; border: 1px solid red;'>❌ Erro 500 - Erro interno do servidor</div>";
    echo "<p>Entre em contato com o suporte da API.</p>";
}

echo "<hr>";

// Links úteis
echo "<h3>🔗 Links Úteis</h3>";
echo "<p><a href='" . admin_url('admin.php?page=wc-pix-gateway-settings') . "'>⚙️ Voltar às Configurações</a></p>";
echo "<p><a href='" . admin_url('admin.php?page=wc-settings&tab=checkout') . "'>💳 Configurações de Pagamento</a></p>";

echo "<hr>";
echo "<p><em>Teste realizado em: " . date('d/m/Y H:i:s') . "</em></p>";
?>
